# Scrobbler.pm 
#
# Copyright (c) SlimScrobbler Team:
#
# Stewart Loving-Gibbard (sloving-gibbard@uswest.net) -- Original author
# Mike Scott (slimscrobbler@hindsight.it) -- SlimServer 5.x changes
# Ian Parkinson (iwp@iwparkinson.plus.com) -- Background submission,
#                                             SlimServer 6.x changes, and
#                                             other tweaks
# Eric Gauthier, Dean Blacketter: !$client fix
# Eric Koldinger: Configuration panel
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.
use strict;
                       
package Plugins::Scrobbler;


# NOTE TO USERS
# You used to have to edit this source file to provide your audioscrobbler
# userid and password. In most cases you should no longer edit this file;
# instead configure the plugin via the Plugins menu on the slimserver
# admin web pages.
# If you use different audioscrobbler accounts for different players,
# you still need to edit the playerAccounts hashtable below; but you must
# still provide a default userid/password on the admin web pages.


# TODO Rationalise the use list. Much work has been offloaded to
# Session.pm and Track.pm, so some of these may no longer be necessary.
use Slim::Player::Playlist;
use Slim::Player::Source;
use Slim::Player::Sync;
use Slim::Utils::Misc;
use Slim::Utils::Timers;
use Slim::Utils::Strings qw (string);
use Slim::Player::Client;
use Slim::Music::Info;
use Time::HiRes;
use MP3::Info;
use Class::Struct;
# For temp directory
use File::Spec;

use FindBin qw($Bin);
use lib(File::Spec->catdir($Bin, 'Plugins','SlimScrobbleSupport'));

use Math::Round;
use LWP::UserAgent;
# For min/max
use List::Util;

use Scrobbler::Session;
use Scrobbler::Track;

# If Perl version less than 5.8, use old Unicode stuff
if ($^V lt v5.8)
{
   # print "Perl version less than 5.8!\n";
   
   # Haven't been able to get this working yet; no Unicode on Perl 5.6 for now.
   # This means non-ASCII characters in song & track titles may be submitted
   # incorrectly. 

   # require Unicode::MapUTF8;
   # import Unicode::MapUTF8;
}
else
{
   #print "Perl version 5.8 or greater.\n";

   # Requires Perl 5.8.0 or greater. Needed for UTF-8 encoding.
   require Encode;
   import Encode;
}

# Requires Perl 5.8.0 or greater. Needed for UTF-8 encoding.
#use Encode;



##################################################
### Set the variables here, Scrobble hackers   ###
##################################################

# Scrobbler allows you to associate Audioscrobbler accounts
# with player IDs. These players could be actual
# SlimDevices hardware units, or streaming clients, so long
# as you reference them by the same names you provide
# in the web server interface.
#
# Any players not specifically referenced in playerAccountswill be
# associated with the default ID, set with the config pages
#
# If you sync players, the slave players will stop recording their
# play data, on the assumption that a synchronized play counts
# as one listen. All tracks listened to while sync'd will be
# recorded by the *MASTER* unit only.

# OPTIONAL : Give any additional accounts you wish to use
#            here, then associate them with player IDs
#            in playerAccounts.

#my $SCROBBLE_USERID_SLIMTEST2 = "slimtest2";
#my $SCROBBLE_PASSWORD_SLIMTEST2 = "ohJC2jmj";

#my $SCROBBLE_USERID_HOUSE = "myhouse";
#my $SCROBBLE_PASSWORD_HOUSE = "ohmigodthereisgreenwaterpouringoutoftheceiling";

# You only need to fill this in if you have more than one
# AudioScrobbler account; see above!

my %playerAccounts = 
(
#   'Dumpy'   => [ $SCROBBLE_USERID_SLIMTEST2, $SCROBBLE_PASSWORD_SLIMTEST2 ],
#   'Living' =>  [ $SCROBBLE_USERID_HOUSE, $SCROBBLE_PASSWORD_HOUSE ], 
);


# 1 sets chatty debug messages to ON
# 0 sets chatty debug messages to OFF
#
# (You will still need to use "slimserver.pl --d_cli" at the command line to see them.)
my $SCROBBLE_DEBUG = Slim::Utils::Prefs::get("scrobbler-debug");

# How many submissions should be queued before we
# attempt a submit to AudioScrobbler. This is only one
# of several factors controlling submission; don't be 
# alarmed if it takes longer than $MAX_PENDING_COUNT
# songs to happen.
#my $MAX_PENDING_COUNT = Slim::Utils::Prefs::get("scrobbler-max-pending");


#################################################
### Global constants - do not change casually ###
#################################################

# There are multiple different conditions which
# influence whether a track is considered played:
#
#  - A minimum number of seconds a track must be 
#    played to be considered a play. Note that
#    if set too high it can prevent a track from
#    ever being noted as played - it is effectively
#    a minimum track length. Overrides other conditions!
#
#  - A percentage play threshold. For example, if 50% 
#    of a track is played, it will be considered played.
#
#  - A time played threshold. After this number of
#    seconds playing, the track will be considered played.


# Identity of this plug-in
my $SCROBBLE_PLUGIN_NAME = "SlimScrobbler";
my $SCROBBLE_PLUGIN_VERSION = "0.30";

my $trackVars = {
    # After this much of a track is played, mark as played.
    SCROBBLE_PERCENT_PLAY_THRESHOLD => .50,

    # After this number of seconds of a track is played, mark as played.
    SCROBBLE_TIME_PLAY_THRESHOLD => 240,

    # The minimum length (in seconds) of tracks allowed by AudioScrobbler
    SCROBBLE_MINIMUM_LENGTH => 30,

    # The maximum length (in seconds) of tracks allowed by AudioScrobbler
    # (30 minutes)
    SCROBBLE_MAXIMUM_LENGTH => 1800,

    # String/character to separate fields on submission
    # TODO this is defined here and in sessionVars; fix it.
    SCROBBLE_SUBMIT_DELIMITER => "&",
};

my $sessionVars = {
# After a submission error, the number of seconds before retrying,
# as used by the non-background submitter
    SCROBBLE_RETRY_DELAY => 1800,

# After a submission error, minimum and maximum number of seconds before
# retrying, as used by the background submitter
# After the first failure, it retries after MIN_DELAY, and doubles
# each time, maxing out at MAX_DELAY.
    SCROBBLE_RETRY_MIN_DELAY => 60,
    SCROBBLE_RETRY_MAX_DELAY => 7200,


# Length of time, in seconds, to wait for a response from AudioScrobbler
# (ignored by the background submitter)
    SCROBBLE_COMMS_TIMEOUT => 10,

# Length of time, in seconds, to wait for certain network operations
# Should be short enough to not cause pauses in playback
    SCROBBLE_NB_COMMS_TIMEOUT => 5,

# Identity of this plug-in
    SCROBBLE_PLUGIN_NAME => $SCROBBLE_PLUGIN_NAME,
    SCROBBLE_PLUGIN_VERSION => $SCROBBLE_PLUGIN_VERSION,

# Base URL to the AudioScrobbler server
    SCROBBLE_SERVER => "post.audioscrobbler.com/",
# URL for testing
#   SCROBBLE_SERVER => "audioscrobbler.sourceforge.net/submissiontest.php",

# User/password for testing with above test URL
#my $TEST_USER = "test";
#my $TEST_PASSWORD = "testpass";

# Client ID for our plugin                                       
    SCROBBLE_CLIENT_ID => "slm",

# String/character to separate fields on submission
    SCROBBLE_SUBMIT_DELIMITER => "&",
};

# Indicator if scrobbler is hooked or not
# 0= No
# 1= Yes
my $SCROBBLE_HOOK = 0;

# Whether we should, by default, use the background submitter.
# Owners of older (SliMP3) players may want to disable this, as it
# can cause occasional breaks while playing music. However, when disabled,
# you'll see occasional lengthy breaks between tracks and (probably) more
# spam warnings from AudioScrobbler. If you do want to disable this, set
# scrobbler-background-submit = 0 in the slimserver.conf file.
my $SCROBBLE_BACKGROUND_DEFAULT = 1;

# Export the version to the server
use vars qw($VERSION);
$VERSION = $SCROBBLE_PLUGIN_VERSION;


##################################################
### SlimServer Plugin API                      ###
##################################################

# This section needs a lot of help. At the time
# that I wrote this, I had no SlimDevices hardware,
# and so couldn't test it.

sub getDisplayName() 
{
# Slim changed this at slimserver v6. Seems an odd thing to change
# to me, given that it must break every plugin out there, and makes it
# impossible to write plugins that support both pre- and post-v6 servers.
#   return string('PLUGIN_SCROBBLER')
    return "PLUGIN_SCROBBLER";
}

sub strings
{
    local $/ = undef;
    <DATA>;
}



sub setMode() 
{
	my $client = shift;
	$client->lines(\&lines);
}


sub enabled() 
{
	my $client = shift;
	return 1;
}

sub initPlugin()
{
    if (isConfigValid()) {
	hookScrobbler();
    }
    else {
	Plugins::Scrobbler::scrobbleMsg("Scrobbler not hooked; please set userid and password in Web UI\n");
    }
}

sub shutdownPlugin()
{
    unHookScrobbler();
}

# I expect this is all wrong.

my %functions = (
	'up' => sub  {
		my $client = shift;
		Slim::Display::Animation::bumpUp($client);
	},
	'down' => sub  {
	    my $client = shift;
		Slim::Display::Animation::bumpDown($client);
	},
	'left' => sub  {
		my $client = shift;
		Slim::Buttons::Common::popModeRight($client);
	},
	'right' => sub  {
		my $client = shift;
		Slim::Display::Animation::bumpRight($client);
	},
	'play' => sub {
		my $client = shift;
		if ( !isConfigValid() ) {
		    # do nothing...
		}
	        elsif ( $SCROBBLE_HOOK == 0 ) {
		  my ($line1, $line2) = (string('PLUGIN_SCROBBLER'), string('PLUGIN_SCROBBLER_ACTIVATED'));
	   	  Plugins::Scrobbler::hookScrobbler();
		  Slim::Display::Animation::showBriefly($client, $line1, $line2);
		} else {
		  my ($line1, $line2) = (string('PLUGIN_SCROBBLER'), string('PLUGIN_SCROBBLER_NOTACTIVATED'));
	   	  Plugins::Scrobbler::unHookScrobbler();
		  Slim::Display::Animation::showBriefly($client, $line1, $line2);
		}
	},
	'add' => sub {
		my $client = shift;
		if (isConfigValid()) {
		    my ($line1, $line2) = (string('PLUGIN_SCROBBLER'), string('PLUGIN_SCROBBLER_SUBMITTING'));
		    Slim::Display::Animation::showBriefly($client, $line1, $line2);
		    getPlayerStatusForClient($client)->session()->attemptToSubmitToAudioScrobbler();
		}
	}
);


sub lines() 
{
	my ($line1, $line2);
	$line1 = string('PLUGIN_SCROBBLER');
	if ( !isConfigValid() ) {
	        $line2 = string('PLUGIN_SCROBBLER_BAD_CONFIG');
	}
	elsif ( $SCROBBLE_HOOK == 0 ) {
		$line2 = string('PLUGIN_SCROBBLER_HIT_PLAY_TO_START');
	}
	else {
		$line2 = string('PLUGIN_SCROBBLER_HIT_PLAY_TO_STOP');
	}
	return ($line1, $line2);
}



##################################################
### Scrobbler per-client Data                  ###
##################################################


# Each client's playStatus structure. 
# Start's empty; as new players appear we add them using getPlayerStatusForClient().
my %playerStatusHash = ();

# Each user's Session object.
# As with playerStatusHash, starts empty. Sessions are added as we need them,
# indexed by userid
my %sessionHash = ();


# The master structure, one per Slim client. Matches the client to an
# AudioScrobbler session and the currently playing track
struct PlayTrackStatus => {

   # Client Name & ID 
   # Used for debugging at the moment
   clientName => '$',
   clientID => '$',

   # The AudioScrobbler Session, which manages per-userid state
   session => 'Scrobbler::Session',

   # The track currently playing. May be undef.
   currentTrack => 'Scrobbler::Track',

   # Is this player on or off? 
   # (HEY SlimDevices staff: The "power" commandCallbck DOES NOT return "on" or "off", so I must
   #  track this myself! Very error-prone, and redundant.)
   isOn => '$',
};


# Get the appropriate user ID & password for the given 
# friendly player name ("Stan", "Dumpy", etc.)
sub getUserIDPasswordForPlayer($)
{
   my $clientName = shift;

   if (!defined($playerAccounts{$clientName}))
   {
      Plugins::Scrobbler::scrobbleMsg("Using default User/password ($clientName)\n");
      return ( Slim::Utils::Prefs::get("scrobbler-user"), Slim::Utils::Prefs::get("scrobbler-password"));
   }
   else
   {
      Plugins::Scrobbler::scrobbleMsg("Found User/password ($clientName)\n");
      return ( $playerAccounts{$clientName}[0], $playerAccounts{$clientName}[1] ) ;
   }
}


# Set the appropriate default values for this playerStatus struct
sub setPlayerStatusDefaults($$$$)
{
   # Parameter - client
   my $client = shift;

   # Parameter - Player status structure.
   # Uses pass-by-reference
   my $playerStatusToSetRef = shift;

   # Parameters -- Client name & ID
   my $clientName = shift;
   my $clientID = shift;

   # Set client name & ID (used for debugging at the moment)
   $playerStatusToSetRef->clientName($clientName);
   $playerStatusToSetRef->clientID($clientID);

   # Get our session object
   $playerStatusToSetRef->session(getSessionForClient($client));

   # Are we on? (If we've heard about the player, it is on..)
   $playerStatusToSetRef->isOn('true');
}

# Obtain a Session for this client; either re-use an existing one
# (indexed by userid) or create a new one.
# If we are using an existing Session, update the password as
# specified by this player. (Though I think the official line should be
# that multiple client definitions with the same userid but differing
# passwords will lead to undefined behaviour; this is an incorrect
# configuration anyway)
sub getSessionForClient($)
{
    my $client = shift;

    # Extract userId and password for the client
    my ($userID, $password) = getUserIDPasswordForPlayer(Slim::Player::Client::name($client));
    Plugins::Scrobbler::scrobbleMsg("User ID: $userID\n");

    if (!defined($sessionHash{$userID}))
    {
	Plugins::Scrobbler::scrobbleMsg("Creating new Session for $userID\n");

	# Create the session
	my $sess=Scrobbler::Session->new();
	$sess->userid($userID);
	$sess->password($password);

        $sessionHash{$userID} = $sess;
        addSessionToSubmitter($sess);
    }
    else
    {
	Plugins::Scrobbler::scrobbleMsg("Reusing existing Session for $userID\n");
	$sessionHash{$userID}->password($password);
    }

    return $sessionHash{$userID};
}


sub setAllPasswords
{
    my $client;
    
    if ($SCROBBLE_HOOK && !isConfigValid()) {
	Plugins::Scrobbler::scrobbleMsg("Configuration invalid; disabling Scrobbler\n");
        unHookScrobbler();
    }

    if (isConfigValid()) {
	Plugins::Scrobbler::scrobbleMsg("Setting all passwords\n");
	foreach $client (Slim::Player::Client::clients())
	{
	    my $playerStatus=getPlayerStatusForClient($client);
	    $playerStatus->session(getSessionForClient($client));
	}
    }

    if (!$SCROBBLE_HOOK && isConfigValid()) {
	Plugins::Scrobbler::scrobbleMsg("Configuration is now valid; enabling Scrobbler\n");
        hookScrobbler();
    }
}


# Get the player state for the given client.
# Will create one for new clients.
sub getPlayerStatusForClient($)
{
   # Parameter - Client structure
   my $client = shift;

   # Get the friendly name for this client
   my $clientName = Slim::Player::Client::name($client);
   # Get the ID (IP) for this client
   my $clientID = Slim::Player::Client::id($client);

   # These messages get pretty tedious when debugging, even for a chatty client.
   #Plugins::Scrobbler::scrobbleMsg("Asking about client $clientName ($clientID)\n");

   # If we haven't seen this client before, create a new per-client 
   # playState structure.
   if (!defined($playerStatusHash{$client}))
   {
      Plugins::Scrobbler::scrobbleMsg("Creating new PlayerStatus for $clientName ($clientID)\n");
      
      # Create new playState structure
      $playerStatusHash{$client} = PlayTrackStatus->new();

      # Set appropriate defaults
      setPlayerStatusDefaults($client, $playerStatusHash{$client}, $clientName, $clientID);
   }
   else
   {
      # These messages get pretty tedious when debugging, even for a chatty client.
      #Plugins::Scrobbler::scrobbleMsg("Already knew about $clientName ($clientID)\n");
   }

   # If it didn't exist, it does now - 
   # return the playerStatus structure for the client.
   return $playerStatusHash{$client};
}


################################################
### AudioScrobbler main routines             ###
################################################


# A wrapper to allow us to uniformly turn on & off Scrobbler debug messages
sub scrobbleMsg($)
{
   # Parameter - Message to be displayed
   my $scrobbleMessage = shift;

   if (getOrDefault("scrobbler-debug", 0) eq 1)
   {
      msg($scrobbleMessage);      
   }
}


# Check whether the default userid/password has been set. Used to keep
# Scrobbler unhooked without valid configuration.
sub isConfigValid()
{
    my $userid = Slim::Utils::Prefs::get("scrobbler-user");
    my $password = Slim::Utils::Prefs::get("scrobbler-password");

    if (!defined($userid) || ($userid eq "")) { return 0; }
    if (!defined($password) || ($password eq "")) { return 0; }
    
    return 1;
}


# Hook the scrobbler to the play events.
# Do this as soon as possible during startup.
# Only call this if config is valid.
sub hookScrobbler()
{  
    if ($SCROBBLE_HOOK == 0) {
	Plugins::Scrobbler::scrobbleMsg("hookScrobbler() engaged, SlimScrobbler V$SCROBBLE_PLUGIN_VERSION activated.\n");
	Slim::Player::Playlist::setExecuteCommandCallback(\&Plugins::Scrobbler::commandCallback);
	$SCROBBLE_HOOK=1;

	if (useBackgroundSubmitter()) {
	    setSubmitterTimer();
	}
    }
    else {
	Plugins::Scrobbler::scrobbleMsg("SlimScrobbler already active, ignoring hookScrobbler() call");
    }
}


# Unhook the Scrobbler's play event callback function. 
# Do this as the plugin shuts down, if possible.
sub unHookScrobbler()
{
   if ($SCROBBLE_HOOK == 1) {
       if (useBackgroundSubmitter()) {
	   cancelSubmitterTimer();
       }

       # Note that CLI just has this as "...(\&commandCallback)";
       # I'm not sure if what I've done is correct.
       Plugins::Scrobbler::scrobbleMsg("unHookScrobbler() engaged, SlimScrobbler V$SCROBBLE_PLUGIN_VERSION deactivated.\n");
       Slim::Player::Playlist::clearExecuteCommandCallback(\&Plugins::Scrobbler::commandCallback);
       $SCROBBLE_HOOK=0;
   }
   else {
       Plugins::Scrobbler::scrobbleMsg("SlimScrobbler not active, ignoring unHookScrobbler() call\n");
   }
}


# These xxxCommand() routines handle commands coming to us
# through the command callback we have hooked into.

sub openCommand($$)
{
   ######################################
   ### Open command
   ######################################

   # This is the chief way we detect a new song being played, NOT the play command.

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Parameter - filename of track being played
   my $filename = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Open command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   # Stop old song, if needed
   if (defined($playStatus->currentTrack()))
   {
      stopTimingSong($playStatus);
   }

   # Get new song data
   my $totalLength;
   my $artistName;
   my $trackTitle;
   my $albumName;
   my $ds = Slim::Music::Info::getCurrentDataStore();
   if ($ds) {
       my $track = $ds->objectForUrl($filename);
       if ($track) {
	   $totalLength = $track->durationSeconds();
	   $artistName  = $track->artist();
	   $trackTitle  = $track->title();

	   my $album = $track->album();
	   if ($album) { $albumName = $album->title(); }
       }
   }

# Preserved for posterity, just in case anybody tries to run against
# a pre-V6 slimserver. If that is you, uncomment these lines and
# comment out the last block of code.
#   my $totalLength = Slim::Music::Info::durationSeconds($filename);
#   my $artistName = Slim::Music::Info::artist($filename);
#   my $trackTitle = Slim::Music::Info::title($filename);
#   my $albumName  = Slim::Music::Info::album($filename);

   # Start timing new song
   startTimingNewSong($playStatus, $filename, $artistName,
                                $trackTitle, $albumName, $totalLength);

   showCurrentVariables($playStatus);
}


sub playCommand($)
{
   ######################################
   ### Play command
   ######################################

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Play command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   my $track = $playStatus->currentTrack();
   if (defined($track)) {
     $track->play();
   }

   # If we just got a play command, but we think the player is off, we need to toggle the
   # player status back to "on" again. (We don't get a 'power' command when someone just hits
   # the play button from a power-off state. Sigh.)
   # We used to lose the currently-playing track if someone did this, but I don't think we do
   # any more.
   if ($playStatus->isOn() eq "false")
   {
      Plugins::Scrobbler::scrobbleMsg("Looks like someone forgot to tell us power was back on for $clientName ($clientID)]...\n");
      # Plugins::Scrobbler::scrobbleMsg("NOTE: Right now we lose track of the current track's play in this circumstance, sorry! ($clientID)]\n");

      $playStatus->isOn("true");
   }
   
   showCurrentVariables($playStatus);
}

sub pauseCommand($$)
{
   ######################################
   ### Pause command
   ######################################

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   # Parameter - Optional second parameter in command
   # (This is for the case <pause 0 | 1> ). 
   # If user said "pause 0" or "pause 1", this will be 0 or 1. Otherwise, undef.
	my $secondParm = shift;

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Pause command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   my $track=$playStatus->currentTrack();
   if (defined($track)) {
      # Just a plain "pause"
      if (!defined($secondParm))
      {
         Plugins::Scrobbler::scrobbleMsg("Vanilla Pause\n");
         $track->togglePause();
      }

      # "pause 1" means "pause true", so pause and stop timing, if not already paused.
      elsif ( ($secondParm eq 1) )
      {
         Plugins::Scrobbler::scrobbleMsg("Pausing (1 case)\n");
         $track->pause();      
      }

      # "pause 0" means "pause false", so unpause and resume timing, if not already timing.
      elsif ( ($secondParm eq 0) )
      {
         Plugins::Scrobbler::scrobbleMsg("Pausing (0 case)\n");
         $track->play();      
      }
   
      else
      {      
         Plugins::Scrobbler::scrobbleMsg("Pause command ignored, assumed redundant.\n");
      }
   }

    showCurrentVariables($playStatus);
}


sub stopCommand($)
{
   ######################################
   ### Stop command
   ######################################

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Stop command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   if (defined($playStatus->currentTrack()))
   {
      stopTimingSong($playStatus);
   }

   showCurrentVariables($playStatus);
}

sub powerOnCommand ($)
{
   ######################################
   ## Power on command
   ######################################

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Power on command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   # Set our on/off flag
   $playStatus->isOn('true');
   
   # I think I've seen times when the power state tracking has got confused,
   # so I'm going to pause any playing song here. The end result will be correct
   # whether we've just turned power off or on; either way, the track is paused.
   my $track=$playStatus->currentTrack();
   if (defined($track)) {
      $track->pause();
   }
   
   showCurrentVariables($playStatus);
}

# We used to treat power-off as a hard stop, but I always find tracks resume
# where they left off after power-on. So we now treat power-off as a pause.
sub powerOffCommand($)
{
   ######################################
   ## Power off command
   ######################################

   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Get name & ID of this player
   my $clientName = $playStatus->clientName();
   my $clientID =   $playStatus->clientID();

   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");
   Plugins::Scrobbler::scrobbleMsg("Power off command [$clientName ($clientID)]\n");
   Plugins::Scrobbler::scrobbleMsg("*----------------------------\n");

   my $track=$playStatus->currentTrack();
   if (defined($track)) {
      $track->pause();
   }

   # Set our on/off flag
   $playStatus->isOn('false');

    showCurrentVariables($playStatus);
}



# This gets called during playback events.
# We look for events we are interested in, and start and stop our various
# timers accordingly.

sub commandCallback($) 
{
   # These are the two passed parameters
   my $client = shift;
   my $paramsRef = shift;

   # Some commands have no client associated with them, so we ignore them.
   return if (!$client); 

   # Get the PlayerStatus
   my $playStatus = getPlayerStatusForClient($client);

   # Debugging a crash
   if (!$playStatus) 
   {
      # Get the friendly name for this client
      my $clientName = Slim::Player::Client::name($client);
      # Get the ID (IP) for this client
      my $clientID = Slim::Player::Client::id($client);

      Plugins::Scrobbler::scrobbleMsg("WARNING: No playStatus for $clientName ($clientID) in commandCallback!\n");
   }

   # I had hoped that this would be accurate enough
   # to use in the place of all this ugly state tracking, but
   # sometimes it says "stop" when the *next* command
   # is about to put it into play mode. 

   # In the end, you need to watch commands anyhow, just
   # a different set of them. So, sticking with original
   # implementation.

   #my $playMode = Slim::Player::Source::playmode($client);
   #Plugins::Scrobbler::scrobbleMsg("[****} Playmode according to Slim Code: $playMode\n");

#   Plugins::Scrobbler::scrobbleMsg("====New commands:\n");
#   foreach my $param (@$paramsRef)
#   {
#      Plugins::Scrobbler::scrobbleMsg("  command: $param\n");
#   }

   # showCurrentVariables($playStatus);

   my $slimCommand = @$paramsRef[0];
   my $paramOne = @$paramsRef[1];

   ######################################
   ### Open command
   ######################################

   # This is the chief way we detect a new song being played, NOT play.

   if ($slimCommand eq "open") 
   {
      my $trackOriginalFilename = $paramOne;

      openCommand($playStatus, $trackOriginalFilename);
   }

   ######################################
   ### Play command
   ######################################

   if( ($slimCommand eq "play") ||
       (($slimCommand eq "mode") && ($paramOne eq "play")) )
   {
      playCommand($playStatus);
   }

   ######################################
   ### Pause command
   ######################################

   if ($slimCommand eq "pause")
   {
      # This second parameter may not exist,
      # and so this may be undef. Routine expects this possibility,
      # so all should be well.
      pauseCommand($playStatus, $paramOne);
   }

   if (($slimCommand eq "mode") && ($paramOne eq "pause"))
   {  
      # "mode pause" will always put us into pause mode, so fake a "pause 1".
      pauseCommand($playStatus, 1);
   }

   ######################################
   ### Sleep command
   ######################################

   if ($slimCommand eq "sleep")
   {
      # Sleep has no effect on streamed players; is this correct for SlimDevices units?
      # I can't test it.

      #Plugins::Scrobbler::scrobbleMsg("===> Sleep activated! Be sure this works!\n");

      #pauseCommand($playStatus, undef());
   }

   ######################################
   ### Stop command
   ######################################

   if ( ($slimCommand eq "stop") ||
        (($slimCommand eq "mode") && ($paramOne eq "stop")) )
   {
      stopCommand($playStatus);
   }

   ######################################
   ### Stop command
   ######################################

   if ( ($slimCommand eq "playlist") && ($paramOne eq "sync") )
   {
      # If this player syncs with another, we treat it as a stop,
      # since whatever it is presently playing (if anything) will end.
      stopCommand($playStatus);
   }


   ######################################
   ## Power command
   ######################################

   # If we received a potential "on"/"off" parameter..
   if ($paramOne) 
   {
      # Power off
      if ( (($slimCommand eq "power") && ($paramOne eq "off")) ||
           (($slimCommand eq "mode") && ($paramOne eq "off")) )
      {
         #Plugins::Scrobbler::scrobbleMsg("===> Power On with explicit \"Off\" parameter\n");
         powerOffCommand($playStatus);
      }
      # Power on
      elsif ( (($slimCommand eq "power") && ($paramOne eq "on")) ||
           (($slimCommand eq "mode") && ($paramOne eq "on")) )
      {
         #Plugins::Scrobbler::scrobbleMsg("===> Power Off with explicit \"On\" parameter\n");
         powerOnCommand($playStatus);
      }
   }
   # Unfortunately, the second parameter is optional, and we often don't get it.
   else
   {
      # Power off
      if ( ($slimCommand eq "power") && ($playStatus->isOn() eq "true") ) 
      {
         #Plugins::Scrobbler::scrobbleMsg("===> Power Off, since playStatus appears to be On.\n");
         powerOffCommand($playStatus);
      }
      # Power on
      elsif ( ($slimCommand eq "power") && ($playStatus->isOn() eq "false") ) 
      {
         #Plugins::Scrobbler::scrobbleMsg("===> Power On, since playStatus appears to be Off.\n");
         powerOnCommand($playStatus);
      }
   }

}


# Create a UTC time string for now
sub getUTCTimeRightNow()
{

#   Using an abbreviated UTC Time format:
#
#   YYYY-MM-DD hh:mm:ss

   my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday);
   
   # Get the time variables
   ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime(time);  
   
   # Adjust for human-readable output 
   $mday = sprintf("%02u", $mday);
   # months numbered from 0;
   $mon += 1;
   $mon = sprintf("%02u", $mon);
   # Year is number of years since 1900; adjust
   $year += 1900;
   # Time
   $hour = sprintf("%02u", $hour);
   $min = sprintf("%02u", $min);
   $sec = sprintf("%02u", $sec);

   # Build the UTC string
   # Year first
   my $UTCString = $year . "-" . $mon . "-" . $mday;
   # Space
   $UTCString .= " ";
   # Time second
   $UTCString .= $hour . ":" . $min . ":" . $sec; 

   # Return the time we built
   return($UTCString);
}


# A new song has begun playing. Reset the current song
# timer and set new Artist and Track.
sub startTimingNewSong($$$$$$)
{
   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   # Parameters: Artist name & Track title
   my $filename = shift;
   my $artistName = shift;
   my $trackTitle = shift;
   my $albumName  = shift;
   my $totalLength = shift;

   if (!$artistName) { $artistName = ""; }
   if (!$trackTitle) { $trackTitle = ""; }
   if (!$albumName) { $albumName = ""; }

   Plugins::Scrobbler::scrobbleMsg("=======================================\n");
   Plugins::Scrobbler::scrobbleMsg("Starting to time \"$artistName - $trackTitle (from $albumName)\"\n");

   my $newTrack = Scrobbler::Track->new(
        filename    => $filename,
        artist      => $artistName,
        album       => $albumName,
        title       => $trackTitle,
        totalLength => $totalLength,
   );
                     
   # Start the new track timing   
   $newTrack->play();

   # Register our track with the background submitter
   if (useBackgroundSubmitter()) {
       addTrackToSubmitter($playStatus->session(), $newTrack);
   }
   
   $playStatus->currentTrack($newTrack);
}


# Stop timing the current song
# (Either stop was hit or we are about to play another one)
sub stopTimingSong($)
{
   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   my $track = $playStatus->currentTrack();
   if (!defined($track))
   {
      msg("Programmer error in Scrobbler::stopTimingSong() - not already timing!\n");   
   }
   else {
      # Attempt to stop the track timing
      $track->cancel();

      if (!useBackgroundSubmitter()) {
	  # If the track was played long enough to count as a listen, it will
	  # be marked 'Ready'
	  if ($track->isReady())
	  {
	      Plugins::Scrobbler::scrobbleMsg("Track was played long enough to count as listen\n");

	      # Log it to Audioscrobbler
	      my $UTCtime = getUTCTimeRightNow();
	      $playStatus->session()->logTrackToAudioScrobblerAsPlayed($track, $UTCtime);
	      $track->markRegistered();

	      # If we're autosubmit, submit now
	      if (getOrDefault('scrobbler-auto-submit', 1)) {
	        $playStatus->session()->tryToSubmitToAudioscrobblerIfNeeded();
	      }

              # We could also log to history at this point as well..
	  }
	  else {
	      Plugins::Scrobbler::scrobbleMsg("Track was NOT played long enough to count as listen\n");
	  }
      }
  }

}


# Debugging routine - shows current variable values for the given playStatus
sub showCurrentVariables($)
{
   # Parameter - PlayTrackStatus for current client
   my $playStatus = shift;

   Plugins::Scrobbler::scrobbleMsg("======= showCurrentVariables() ========\n");
   
   # Call Track to display artist, track name, etc.
   if (defined($playStatus->currentTrack())) {
     $playStatus->currentTrack()->showCurrentVariables();
   }
   else {
     Plugins::Scrobbler::scrobbleMsg("No track currently timing\n");
   }
   
   my $tmpIsPowerOn = $playStatus->isOn();
   Plugins::Scrobbler::scrobbleMsg("Is power on? : $tmpIsPowerOn\n");

   # Call Session to display pending count, interval, etc.
   $playStatus->session()->showCurrentVariables();

   Plugins::Scrobbler::scrobbleMsg("=======================================\n");
}


sub getFunctions() 
{
	return \%functions;
}

sub sessionVars()
{
    return $sessionVars;
}

sub trackVars()
{
    return $trackVars;
}

########################################################################
## Background submission functions and data
##

my @tracks = ();
my @sessions = ();

sub submitter
{
    my @toDelete = ();

    # First, go through the tracks looking for any to register or delete
    my $i=0;
    my $session;
    my $track;
    foreach my $p (@tracks) {
	$session=$p->{session};
	$track=$p->{track};

	if ($track->isReady()) {
	    Plugins::Scrobbler::scrobbleMsg("Following track is ready to submit:\n");
	    $track->showCurrentVariables();

	    # Register it
	    my $UTCtime = getUTCTimeRightNow();
	    $session->logTrackToAudioScrobblerAsPlayed($track, $UTCtime);
	    $track->markRegistered();

	    push(@toDelete, $i);
	}
	elsif ($track->isRegistered()) {
	    # Probably shouldn't happen, means the track has already
	    # been registered. Oh well, delete it
	    Plugins::Scrobbler::scrobbleMsg("Fogetting about following track:\n");
	    $track->showCurrentVariables();

	    push(@toDelete, $i);
	}
	elsif ($track->isCancelled()) {
	    # We can forget about this track now

	    Plugins::Scrobbler::scrobbleMsg("Forgetting about following track:\n");
	    $track->showCurrentVariables();

	    push(@toDelete, $i);
	}

	$i=$i+1;
    }

    # Delete those marked for deletion (I suspect this wouldn't be a good
    # idea during the previous foreach)
    my $offset=0;
    foreach my $d (@toDelete) {
	splice(@tracks, $d - $offset, 1);
	$offset=$offset+1;
    }


    # Now pass over the sessions, see if they need to submit
    if (getOrDefault('scrobbler-auto-submit', 1)) {
	foreach $session (@sessions) {
	    $session->tryToBackgroundSubmitToAudioscrobblerIfNeeded();
	}
    }
   
    setSubmitterTimer();
}

sub setSubmitterTimer()
{
  Slim::Utils::Timers::setTimer("SlimScrobbler", Time::HiRes::time() + 10,
				\&submitter);
}

sub cancelSubmitterTimer()
{
    Slim::Utils::Timers::killOneTimer("SlimScrobbler", \&submitter);

    # Drop any tracks currently being timed. Otherwise, should the Plugin
    # become active again later, the track will be submitted regardless of
    # whether it was left playing for long enough.
    foreach my $p (@tracks) {
	my $track=$p->{track};
	$track->cancel();
    }
}

sub addTrackToSubmitter($$)
{
    my $session=shift;
    my $track=shift;

    my $p = {
	session => $session,
	track => $track,
    };

    if (!$track->isCancelled()) {
	push(@tracks, $p);
    }
}

sub addSessionToSubmitter($)
{
    push(@sessions, shift);
}

sub useBackgroundSubmitter
{
    return getOrDefault('scrobbler-background-submit', $SCROBBLE_BACKGROUND_DEFAULT);
}


########################################################################
## Interface functions
##

sub setupGroup
{
	my %group = (
		PrefOrder => ['scrobbler-user',
					  'scrobbler-password',
					  'scrobbler-auto-submit',
					  'scrobbler-max-pending'],
		PrefsInTable => 1,
		GroupHead => string('PLUGIN_SCROBBLER_HEADER'),
		GroupDesc => string('PLUGIN_SCROBBLER_DESC'),
		GroupLine => 1,
		GroupSub => 1,
		Suppress_PrefSub => 1,
		Suppress_PrefLine => 1,
		Suppress_PrefHead => 1
	);

	my %prefs = (
		'scrobbler-user' => {
			'validate' => \&Slim::Web::Setup::validateAcceptAll,
			'PrefChoose' => string('PLUGIN_SCROBBLER_USERNAME'),
			'changeIntro' => string('PLUGIN_SCROBBLER_USERNAME'),
			'onChange' => sub { setAllPasswords(); }
		},
		'scrobbler-password' => {
			'validate' => \&Slim::Web::Setup::validateAcceptAll,
			'PrefChoose' => string('PLUGIN_SCROBBLER_PASSWORD'),
			'changeIntro' => string('PLUGIN_SCROBBLER_PASSWORD'),
			'changeMsg' => string('PLUGIN_SCROBBLER_PASSWORD_CHANGED'),
			'inputTemplate' => 'setup_input_passwd.html',
			'onChange' => sub { setAllPasswords(); }
		},
		'scrobbler-auto-submit' => {
			'validate' => \&Slim::Web::Setup::validateTrueFalse ,
			'PrefChoose' => string('PLUGIN_SCROBBLER_AUTOSUBMIT'),
                        'changeIntro' => string('PLUGIN_SCROBBLER_AUTOSUBMIT_2'),
			'options' =>
				{
				'1' => string('ON'),
				'0' => string('OFF')
				},
			'currentValue' =>
			    sub {return getOrDefault('scrobbler-auto-submit', 1);}
		},
		'scrobbler-max-pending' => {
			'validate' => \&Slim::Web::Setup::validateInt ,
			'validateArgs' => [1, undef, 1, 0],
			'PrefChoose' => string('PLUGIN_SCROBBLER_MAX_PENDING'),
                        'changeIntro' => string('PLUGIN_SCROBBLER_MAX_PENDING_2'),
			'rejectIntro' => 'Invalid Max Pending',
			'rejectMsg' => 'Max Pending Must be a positive number (>= 0)',
			'currentValue' => 
				sub {return getOrDefault('scrobbler-max-pending', 1);}
		}
	);

	return (\%group, \%prefs);
}

sub getOrDefault
{
	my ($pref, $defValue) = @_;
	return $defValue if (!defined(Slim::Utils::Prefs::get($pref)));
	return Slim::Utils::Prefs::get($pref);
}

# Disabled this for now. SlimServer should automatically start the plugin
# via initPlugin. The user now must configure userid/password in the UI.
# TODO DELETE THIS
# This will start the Scrobbler plug-in as soon as it is loaded, without
# any activation in the UI.
#{
#     if ( $SCROBBLE_HOOK ) 
#     { 
#       Plugins::Scrobbler::hookScrobbler();
#     }
#}


1;

__DATA__

PLUGIN_SCROBBLER
	EN	Audioscrobbler Submitter

PLUGIN_SCROBBLER_ACTIVATED
	EN	AudioScrobbler Activated...

PLUGIN_SCROBBLER_NOTACTIVATED
	EN	AudioScrobbler Not Activated...

PLUGIN_SCROBBLER_HIT_PLAY_TO_START
	EN	Hit play to start AudioScrobbler...
	
PLUGIN_SCROBBLER_HIT_PLAY_TO_STOP
	EN	Hit play to stop AudioScrobbler...
	
PLUGIN_SCROBBLER_SUBMITTING
	EN	Submitting data to AudioScrobbler...

PLUGIN_SCROBBLER_HEADER
	EN	Audioscrobbler

PLUGIN_SCROBBLER_DESC
	EN	Chose your settings for the SlimScrobbler

PLUGIN_SCROBBLER_USERNAME
	EN	Audioscrobbler Username 

PLUGIN_SCROBBLER_PASSWORD
	EN	Audioscrobbler Password 

PLUGIN_SCROBBLER_PASSWORD_CHANGED
	EN	Password Changed

PLUGIN_SCROBBLER_AUTOSUBMIT
	EN	AutoSubmit 

PLUGIN_SCROBBLER_AUTOSUBMIT_2
	EN	Audioscrobbler - AutoSubmit 

PLUGIN_SCROBBLER_MAX_PENDING
	EN	Max Pending Requests 

PLUGIN_SCROBBLER_MAX_PENDING_2
	EN	Audioscrobbler - Max Pending Requests

PLUGIN_SCROBBLER_BAD_CONFIG
	EN	Please set username and password

